/**
* \file: gentexgstbuf.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* Implementation of generating texture and gst buffer
*
* \component: gentexgstbuf
*
* \author: Jens Georg <jgeorg@de.adit-jv.com>, Anil V <anil.valmiki@in.bosch.com>
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#ifndef GENTEXGSTBUF_H
#define GENTEXGSTBUF_H

#include <map>
#include <sstream>
#include <gst/gst.h>

/*
 * GStreamer forward declarations
 */
typedef struct _GstBuffer GstBuffer;

namespace generateTexGstbuf {

class BufferTraits
{
public:
    BufferTraits()
    :red(8),
     blue(8),
     green(8),
     alpha(8),
     config(){}

    // buffer depths, 0 equals off.
    uint red;
    uint blue;
    uint green;
    uint alpha;

    typedef std::map<std::string, std::string> Config;
    Config config;
};

class Pool_alloc;
class Texture;

class GenTexGstbuf
{
public:

    std::map<int,Texture *> tex_map;

    /**
     * @brief GenTexGstbuf constructor set Buffertraits and
     * set texture pool size
     * @param BufferTraits
     */
    GenTexGstbuf(const BufferTraits &bufTraits = BufferTraits());

    /**
     * @brief set the size of the Texture
     * @param width new width ,height new height
     * @note Width will be aligned to be a multiple of 16 in the range of
     */
    void setTexSize(uint width, uint height);

    /**
     * @brief get the size of the Texture
     * @return width new width ,height new height after alignment
     */
    void getTexSize(uint *width, uint *height);

    /**
     * @brief nextTex get texture from the pool
     * @return Texture Id
     */
    int nextTex(int time);

    /**
     * @brief releaseTex put texture back to pool,
     * Release meta data of gst buffer bind to a texture
     * @param NULL
     */
    void releaseTex(uint tex_id);

    /**
     * @brief getGstBufferFromTex get gst buffer from texture
     * @param GenTexGstbuf::Texture
     */
    GstBuffer *getGstBufferFromTex(uint tex_id);

    /**
     * @brief get the height of the Texture
     * @return height
     */
    uint getTexHeight();

    /**
     * @brief get the width of the Texture
     * @return width
     */
    uint getTexWidth();

    /**
     * @brief get the depth of the Texture
     * @return depth
     */
    uint getDepth();

    /**
     * @brief get the format of the Texture
     * @return format
     */
    GLuint getTextureFormat();

#if GST_CHECK_VERSION(1,0,0)
    /**
     * @brief get the drm file descriptor
     * @return format
     */
    int getDrm_fd();
#endif

    /**
     * @brief get the size of one buffer in bytes
     * @return get the size of one buffer in bytes
     */
    size_t getBufferSize() { return getTexWidth() * getTexHeight() * (getDepth() / 8); }

    virtual ~GenTexGstbuf();

private:
    uint texWidth;
    uint texHeight;

    BufferTraits _bufTraits;
    Pool_alloc* p_alloc;
    GLuint _textureFormat;
    bool check_size;

#if GST_CHECK_VERSION(1,0,0)
    int drm_fd;

    void setDrm_fd(int);
#endif

    /**
     * @brief get the current texture configuration
     * @return reference to the currently set texture configuration.
     */
    const BufferTraits& getBufferTraits() const;
    void setBufferTraits(const BufferTraits &traits);
    void check_texture_size();
};
} // namespace generateTexGstbuf
#endif // GENTEXGSTBUF_H
